package framework;

import javax.swing.*;
import java.awt.*;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

public abstract class Game extends JFrame {

    private JPanel drawingSurface;
    private Thread windowThread;

    private boolean running;

    private boolean[] pressedKeyMap;
    private boolean[] pressedMouseKeyMap;

    private int mX, mY;

    public Game(String title, int w, int h) {
        super(title);
        this.setSize(w, h);

        pressedKeyMap = new boolean[1024];
        pressedMouseKeyMap = new boolean[4];

        this.windowThread = new Thread(() -> {
            loadGame();
            while (running) {
                updateGame();
                this.drawingSurface.repaint();
            }
        });

        this.drawingSurface
                = new JPanel() {

            @Override
            public void paint(Graphics g) {
                super.paint(g);
                renderGame(g);
            }
        };

        this.drawingSurface.setSize(this.getSize());
        this.add(this.drawingSurface);

        this.addKeyListener(new KeyAdapter() {
            @Override
            public void keyPressed(KeyEvent e) {
                super.keyPressed(e);

                pressedKeyMap[e.getKeyCode()] = true;
            }

            @Override
            public void keyReleased(KeyEvent e) {
                super.keyReleased(e);

                pressedKeyMap[e.getKeyCode()] = false;
            }

        });

        this.addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(MouseEvent e) {
                super.mousePressed(e);
                mX = e.getX();
                mY = e.getY();
                pressedMouseKeyMap[e.getButton()] = true;
            }

            @Override
            public void mouseReleased(MouseEvent e) {
                super.mouseReleased(e);
                pressedMouseKeyMap[e.getButton()] = false;
            }

        });

        this.setDefaultCloseOperation(EXIT_ON_CLOSE);


    }

    public boolean isKeyDown(int key) {
        return pressedKeyMap[key];
    }

    public boolean isKeyUp(int key) {
        return !isKeyDown(key);
    }

    public boolean isMouseKeyDown(int key) {
        return pressedMouseKeyMap[key];
    }

    public boolean isMouseKeyUp(int key) {
        return !isMouseKeyDown(key);
    }

    public int getMouseX() {
        return mX;
    }

    public int getMouseY() {
        return mY;
    }

    public final void run() {
        this.setVisible(true);
        this.running = true;

        this.windowThread
                .start();
    }

    public abstract void loadGame();

    public abstract void updateGame();

    public abstract void renderGame(Graphics g);

}
